<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\LineaInvestigacion;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Inertia\Inertia;

class AdminLineaInvestigacionController extends Controller
{
    /**
     * GET /admin/lineas
     * Listado con filtros, orden y paginación (similar a AdminProyectoInvestigacionController@index).
     */
    public function index(Request $request)
    {
        $q        = $request->string('q')->toString();
        $tipo     = $request->string('tipo')->toString();        // 'Prioritaria' | 'Linea'
        $prof     = $request->string('profesor')->toString();    // like
        $activo   = $request->string('activo')->toString();      // '1' | '0' | ''
        $year     = $request->integer('year');                   // año exacto de la fecha
        $yearFrom = $request->integer('year_from');
        $yearTo   = $request->integer('year_to');
        $trash    = $request->string('trash', 'none')->toString(); // 'none' | 'with' | 'only'

        $perPage  = (int) max(5, min(100, $request->integer('per_page', 12)));
        $sort     = $request->string('sort', 'nombre')->toString(); // 'nombre'|'profesor'|'fecha'|'tipo'|'created_at'
        $dir      = $request->string('dir', 'asc')->toString();     // 'asc'|'desc'

        $builder = LineaInvestigacion::query();

        // Papelera / SoftDeletes
        if ($trash === 'with') {
            $builder->withTrashed();
        } elseif ($trash === 'only') {
            $builder->onlyTrashed();
        }

        // Búsqueda general (nombre, profesor, descripcion, codigo, uid)
        $builder->when($q, fn($qb) => $qb->search($q));

        // Filtros específicos
        $builder->when($tipo,   fn($qb) => $qb->where('tipo', $tipo));
        $builder->when($prof,   fn($qb) => $qb->where('profesor', 'like', "%{$prof}%"));
        if ($activo !== '') {
            $builder->where('activo', $activo === '1');
        }

        // Filtro por año exacto (a partir de 'fecha' date)
        if ($year) {
            $builder->whereYear('fecha', $year);
        }

        // Rango de años
        if ($yearFrom && $yearTo) {
            $builder->whereBetween(\DB::raw('YEAR(fecha)'), [$yearFrom, $yearTo]);
        } elseif ($yearFrom) {
            $builder->whereYear('fecha', '>=', $yearFrom);
        } elseif ($yearTo) {
            $builder->whereYear('fecha', '<=', $yearTo);
        }

        // Orden permitido
        $allowedSorts = ['nombre','profesor','fecha','tipo','created_at'];
        if (!in_array($sort, $allowedSorts, true)) {
            $sort = 'nombre';
        }
        $builder->orderBy($sort, $dir === 'desc' ? 'desc' : 'asc');

        // Paginación + mapeo para el frontend
        $lineas = $builder
            ->paginate($perPage)
            ->withQueryString()
            ->through(function (LineaInvestigacion $l) {
                return [
                    'id'          => $l->id,
                    'uid'         => $l->uid,
                    'slug'        => $l->slug,
                    'nombre'      => $l->nombre,
                    'profesor'    => $l->profesor,
                    'codigo'      => $l->codigo,
                    'fecha'       => $l->fecha?->toDateString(),
                    'tipo'        => $l->tipo,
                    'descripcion' => $l->descripcion,
                    'activo'      => (bool) $l->activo,
                    'deleted_at'  => $l->deleted_at?->toISOString(),
                    'created_at'  => $l->created_at?->toISOString(),
                    'updated_at'  => $l->updated_at?->toISOString(),
                ];
            });

        // Datos para selects dinámicos (catálogos)
        $tipos = [LineaInvestigacion::TIPO_PRIORITARIA, LineaInvestigacion::TIPO_LINEA];

        // Catálogo de años (puedes reemplazar por un query a min/max de la tabla)
        $yearsCatalog = range((int)date('Y') - 10, (int)date('Y') + 1);
        rsort($yearsCatalog);
        $yearsCatalog = array_map('strval', $yearsCatalog);

        return Inertia::render('Admin/Lineas/Index', [
            'filters' => [
                'q'         => $q,
                'tipo'      => $tipo,
                'profesor'  => $prof,
                'activo'    => $activo,
                'year'      => $year,
                'year_from' => $yearFrom,
                'year_to'   => $yearTo,
                'trash'     => $trash,
                'perPage'   => $perPage,
                'sort'      => $sort,
                'dir'       => $dir,
            ],
            'tipos'  => $tipos,
            'years'  => $yearsCatalog,
            'lineas' => $lineas,
        ]);
    }

    /**
     * POST /admin/lineas
     * Crear una nueva línea de investigación.
     */
    public function store(Request $request)
    {
        $data = $request->validate([
            'uid'         => ['required','string','max:100','unique:lineas_investigacion,uid'],
            'slug'        => ['nullable','string','max:150','unique:lineas_investigacion,slug'],
            'nombre'      => ['required','string','max:255'],
            'profesor'    => ['required','string','max:255'],
            'codigo'      => ['required','string','max:100','unique:lineas_investigacion,codigo'],
            'fecha'       => ['required','date'],
            'tipo'        => ['required','string', Rule::in([LineaInvestigacion::TIPO_PRIORITARIA, LineaInvestigacion::TIPO_LINEA])],
            'descripcion' => ['nullable','string'],
            'activo'      => ['sometimes','boolean'],
        ]);

        // Si no viene 'activo', por defecto true
        $data['activo'] = $data['activo'] ?? true;

        // Trazabilidad (si usas auth)
        if (auth()->check()) {
            $data['created_by'] = auth()->id();
            $data['updated_by'] = auth()->id();
        }

        LineaInvestigacion::create($data);

        return back()->with('success', 'Línea creada correctamente.');
    }

    /**
     * PATCH /admin/lineas/{linea}
     * Actualizar una línea de investigación.
     */
    public function update(Request $request, LineaInvestigacion $linea)
    {
        $data = $request->validate([
            'uid'         => ['required','string','max:100', Rule::unique('lineas_investigacion','uid')->ignore($linea->id)],
            'slug'        => ['nullable','string','max:150', Rule::unique('lineas_investigacion','slug')->ignore($linea->id)],
            'nombre'      => ['required','string','max:255'],
            'profesor'    => ['required','string','max:255'],
            'codigo'      => ['required','string','max:100', Rule::unique('lineas_investigacion','codigo')->ignore($linea->id)],
            'fecha'       => ['required','date'],
            'tipo'        => ['required','string', Rule::in([LineaInvestigacion::TIPO_PRIORITARIA, LineaInvestigacion::TIPO_LINEA])],
            'descripcion' => ['nullable','string'],
            'activo'      => ['sometimes','boolean'],
        ]);

        if (auth()->check()) {
            $data['updated_by'] = auth()->id();
        }

        $linea->update($data);

        return back()->with('success', 'Línea actualizada correctamente.');
    }

    /**
     * DELETE /admin/lineas/{linea}
     * Enviar a papelera (soft delete).
     */
    public function destroy(LineaInvestigacion $linea)
    {
        $linea->delete();
        return back()->with('success', 'Línea enviada a la papelera.');
    }

    /**
     * PATCH /admin/lineas/{id}/restore
     * Restaurar una línea eliminada.
     */
    public function restore($id)
    {
        $linea = LineaInvestigacion::withTrashed()->findOrFail($id);
        if ($linea->trashed()) {
            $linea->restore();
        }

        return back()->with('success', 'Línea restaurada.');
    }

    /**
     * DELETE /admin/lineas/{id}/force
     * Eliminación definitiva.
     */
    public function forceDestroy($id)
    {
        $linea = LineaInvestigacion::withTrashed()->findOrFail($id);
        $linea->forceDelete();

        return back()->with('success', 'Línea eliminada definitivamente.');
    }
}
