<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Laboratorio;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Storage;
use Inertia\Inertia;

class AdminLaboratorioController extends Controller
{
    /**
     * GET /admin/laboratorios
     * Filtros: q, trash(none|with|only), per_page, sort(nombre|created_at), dir(asc|desc)
     */
    public function index(Request $request)
    {
        $q       = $request->string('q')->toString();
        $trash   = $request->string('trash', 'none')->toString();  // 'none' | 'with' | 'only'
        $perPage = (int) max(5, min(100, $request->integer('per_page', 10)));
        $sort    = $request->string('sort', 'nombre')->toString(); // 'nombre'|'created_at'
        $dir     = $request->string('dir', 'asc')->toString();     // 'asc'|'desc'

        $builder = Laboratorio::query();

        // Papelera
        if ($trash === 'with') {
            $builder->withTrashed();
        } elseif ($trash === 'only') {
            $builder->onlyTrashed();
        }

        // Búsqueda: nombre, responsable, descripción, ubicacion(edificio/piso/oficina), asignaturas
        $builder->when($q, function ($query, $q) {
            $query->where(function ($q2) use ($q) {
                $q2->where('nombre', 'like', "%{$q}%")
                   ->orWhere('responsable', 'like', "%{$q}%")
                   ->orWhere('descripcion', 'like', "%{$q}%")
                   ->orWhere('slug', 'like', "%{$q}%")
                   // JSON search (simple LIKE sobre el JSON serializado)
                   ->orWhere('ubicacion->edificio', 'like', "%{$q}%")
                   ->orWhere('ubicacion->piso', 'like', "%{$q}%")
                   ->orWhere('ubicacion->oficina', 'like', "%{$q}%")
                   ->orWhereJsonContains('asignaturas', $q); // si coincide exacto con un item
            });
        });

        // Orden
        if (!in_array($sort, ['nombre','created_at'], true)) {
            $sort = 'nombre';
        }
        $builder->orderBy($sort, $dir === 'desc' ? 'desc' : 'asc');

        $labs = $builder
            ->paginate($perPage)
            ->withQueryString()
            ->through(function (Laboratorio $lab) {
                return [
                    'id'          => $lab->id,
                    'slug'        => $lab->slug,
                    'nombre'      => $lab->nombre,
                    'responsable' => $lab->responsable,
                    'descripcion' => $lab->descripcion,
                    'ubicacion'   => $lab->ubicacion,
                    'asignaturas' => $lab->asignaturas,
                    'imagenes'    => $lab->imagenes,
                    'deleted_at'  => $lab->deleted_at?->toISOString(),
                    'created_at'  => $lab->created_at?->toISOString(),
                    'updated_at'  => $lab->updated_at?->toISOString(),
                ];
            });

        return Inertia::render('Admin/Laboratorios/Index', [
            'filters' => compact('q','trash','perPage','sort','dir'),
            'labs'    => $labs,
        ]);
    }

    /**
     * POST /admin/laboratorios
     * Crear laboratorio con múltiples imágenes (imagenes_files[]).
     */
    public function store(Request $request)
    {
        $data = $request->validate([
            'slug'        => ['required','string','max:120', 'alpha_dash', Rule::unique('laboratorios','slug')->whereNull('deleted_at')],
            'nombre'      => ['required','string','max:255'],
            'responsable' => ['nullable','string','max:255'],
            'descripcion' => ['nullable','string','max:10000'],
            'ubicacion'   => ['nullable','array'], // { edificio, piso, oficina }
            'ubicacion.edificio' => ['nullable','string','max:50'],
            'ubicacion.piso'     => ['nullable','string','max:50'],
            'ubicacion.oficina'  => ['nullable','string','max:50'],
            'asignaturas' => ['nullable','array'],
            'asignaturas.*' => ['string','max:255'],
            'imagenes'    => ['nullable','array'],    // rutas que ya vengan (opcional)
            'imagenes.*'  => ['string','max:1000'],
            'imagenes_files'   => ['nullable','array'],
            'imagenes_files.*' => ['image','max:4096'], // 4MB c/u
        ]);

        // Subir nuevas imágenes
        $newPaths = [];
        if ($request->hasFile('imagenes_files')) {
            foreach ($request->file('imagenes_files') as $file) {
                $p = $file->store('labs', 'public'); // storage/app/public/labs
                $newPaths[] = "/storage/{$p}";
            }
        }

        $data['imagenes']    = array_values(array_filter(array_merge($data['imagenes'] ?? [], $newPaths)));
        $data['asignaturas'] = $data['asignaturas'] ?? [];
        // limpiar campo temporal
        unset($data['imagenes_files']);

        Laboratorio::create($data);

        return back()->with('success', 'Laboratorio creado correctamente.');
    }

    /**
     * PATCH /admin/laboratorios/{laboratorio}
     * Actualizar datos y manejar galería:
     * - 'imagenes_keep' (array de rutas) para conservar
     * - 'imagenes_files[]' (nuevas imágenes para agregar)
     */
    public function update(Request $request, Laboratorio $laboratorio)
    {
        $data = $request->validate([
            'slug'        => ['required','string','max:120','alpha_dash', Rule::unique('laboratorios','slug')->whereNull('deleted_at')->ignore($laboratorio->id)],
            'nombre'      => ['required','string','max:255'],
            'responsable' => ['nullable','string','max:255'],
            'descripcion' => ['nullable','string','max:10000'],
            'ubicacion'   => ['nullable','array'],
            'ubicacion.edificio' => ['nullable','string','max:50'],
            'ubicacion.piso'     => ['nullable','string','max:50'],
            'ubicacion.oficina'  => ['nullable','string','max:50'],
            'asignaturas' => ['nullable','array'],
            'asignaturas.*' => ['string','max:255'],

            // Galería
            'imagenes_keep'     => ['nullable','array'],  // rutas a conservar
            'imagenes_keep.*'   => ['string','max:1000'],
            'imagenes_files'    => ['nullable','array'],  // nuevas
            'imagenes_files.*'  => ['image','max:4096'],
        ]);

        $keep = $data['imagenes_keep'] ?? [];
        $actual = $laboratorio->imagenes ?? [];

        // Elimina físicamente las imágenes que NO están en keep (solo si apuntan a /storage/labs)
        $toDelete = array_diff($actual, $keep);
        foreach ($toDelete as $path) {
            if (is_string($path) && str_starts_with($path, '/storage/labs/')) {
                Storage::disk('public')->delete(str_replace('/storage/', '', $path));
            }
        }

        // Agregar nuevas imágenes
        $newPaths = [];
        if ($request->hasFile('imagenes_files')) {
            foreach ($request->file('imagenes_files') as $file) {
                $p = $file->store('labs', 'public');
                $newPaths[] = "/storage/{$p}";
            }
        }

        $imagenesFinal = array_values(array_filter(array_merge($keep, $newPaths)));

        $payload = [
            'slug'        => $data['slug'],
            'nombre'      => $data['nombre'],
            'responsable' => $data['responsable'] ?? null,
            'descripcion' => $data['descripcion'] ?? null,
            'ubicacion'   => $data['ubicacion'] ?? null,
            'asignaturas' => $data['asignaturas'] ?? [],
            'imagenes'    => $imagenesFinal,
        ];

        $laboratorio->update($payload);

        return back()->with('success', 'Laboratorio actualizado correctamente.');
    }

    /**
     * DELETE /admin/laboratorios/{laboratorio}
     * Soft delete.
     */
    public function destroy(Laboratorio $laboratorio)
    {
        $laboratorio->delete();

        return back()->with('success', 'Laboratorio enviado a la papelera.');
    }

    /**
     * PATCH /admin/laboratorios/{id}/restore
     * Restaura un laboratorio eliminado.
     */
    public function restore($id)
    {
        $lab = Laboratorio::withTrashed()->findOrFail($id);
        if ($lab->trashed()) {
            $lab->restore();
        }

        return back()->with('success', 'Laboratorio restaurado.');
    }

    /**
     * DELETE /admin/laboratorios/{id}/force
     * Eliminación definitiva (también borra archivos físicos).
     */
    public function forceDestroy($id)
    {
        $lab = Laboratorio::withTrashed()->findOrFail($id);

        // Elimina físicamente las imágenes
        foreach (($lab->imagenes ?? []) as $path) {
            if (is_string($path) && str_starts_with($path, '/storage/labs/')) {
                Storage::disk('public')->delete(str_replace('/storage/', '', $path));
            }
        }

        $lab->forceDelete();

        return back()->with('success', 'Laboratorio eliminado definitivamente.');
    }
}
