<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Docente;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Password;
use Illuminate\Support\Str;
use Inertia\Inertia;

class AdminDocenteController extends Controller
{
    /**
     * GET /admin/docentes
     */
    public function index(Request $request)
    {
        $q        = $request->string('q')->toString();
        $area     = $request->string('area')->toString();
        $trash    = $request->string('trash', 'none')->toString();
        $perPage  = (int) max(5, min(100, $request->integer('per_page', 10)));
        $sort     = $request->string('sort', 'nombre')->toString();
        $dir      = $request->string('dir', 'asc')->toString();

        $builder = Docente::query();

        // Papelera
        if ($trash === 'with') $builder->withTrashed();
        elseif ($trash === 'only') $builder->onlyTrashed();

        // Búsqueda general
        $builder->when($q, function ($query, $q) {
            $query->where(function ($q2) use ($q) {
                $q2->where('nombre', 'like', "%{$q}%")
                   ->orWhere('email', 'like', "%{$q}%")
                   ->orWhere('area', 'like', "%{$q}%");
            });
        });

        // Filtro área
        $builder->when($area, fn($query) => $query->where('area', $area));

        // Orden
        if (!in_array($sort, ['nombre','email','area','created_at'], true)) {
            $sort = 'nombre';
        }
        $builder->orderBy($sort, $dir === 'desc' ? 'desc' : 'asc');

        // Paginación
        $docentes = $builder
            ->paginate($perPage)
            ->withQueryString()
            ->through(function (Docente $d) {
                return [
                    'id'          => $d->id,
                    'nombre'      => $d->nombre,
                    'grado'       => $d->grado,
                    'area'        => $d->area,
                    'foto'        => $d->foto,
                    'email'       => $d->email,
                    'asignaturas' => $d->asignaturas,
                    'enlaces'     => $d->enlaces,
                    'deleted_at'  => $d->deleted_at?->toISOString(),
                    'created_at'  => $d->created_at?->toISOString(),
                    'updated_at'  => $d->updated_at?->toISOString(),
                ];
            });

        // Áreas únicas para filtros
        $areas = Docente::select('area')
            ->whereNotNull('area')
            ->distinct()
            ->pluck('area')
            ->sort()
            ->values();

        return Inertia::render('Admin/Docentes/Index', [
            'filters'  => compact('q','area','trash','perPage','sort','dir'),
            'areas'    => $areas,
            'docentes' => $docentes,
        ]);
    }

    /**
     * POST /admin/docentes
     * Crea un docente y su usuario asociado.
     */
    public function store(Request $request)
    {
        $data = $request->validate([
            'nombre'       => ['required','string','max:255'],
            'grado'        => ['nullable','string','max:255'],
            'area'         => ['nullable','string','max:255'],
            'email'        => ['required','email','max:255', Rule::unique('docentes','email')->whereNull('deleted_at')],
            'asignaturas'  => ['nullable','array'],
            'enlaces'      => ['nullable','array'],
            'foto_file'    => ['nullable','image','max:2048'],
        ]);

        $data['asignaturas'] = $data['asignaturas'] ?? [];
        $data['enlaces'] = $data['enlaces'] ?? [];

        // Procesar imagen si existe
        if ($request->hasFile('foto_file')) {
            $path = $request->file('foto_file')->store('docentes', 'public');
            $data['foto'] = "/storage/{$path}";
        }
        unset($data['foto_file']);

        // Crear docente
        $docente = Docente::create($data);

        // 🔹 Crear usuario asociado con rol Docente (si no existe ya)
        if (!User::where('email', $data['email'])->exists()) {

            $user = User::firstOrCreate(
                ['email' => $data['email']],
                [
                    'name'      => $data['nombre'],
                    'role'      => 'Docente',
                    'is_active' => true,
                    'password'  => Hash::make(Str::password(12)), // temporal (no se usa)
                ]
            );
        }

// Crear usuario asociado si no existe

// Enviar enlace de restablecimiento/definición de contraseña
Password::sendResetLink(['email' => $user->email]);

        return back()->with('success', 'Docente y usuario creados correctamente.');
    }

    /**
     * PATCH /admin/docentes/{docente}
     */
    public function update(Request $request, Docente $docente)
    {
        $data = $request->validate([
            'nombre'       => ['required','string','max:255'],
            'grado'        => ['nullable','string','max:255'],
            'area'         => ['nullable','string','max:255'],
            'foto'         => ['nullable','string','max:500'],
            'email'        => ['required','email','max:255', Rule::unique('docentes','email')->whereNull('deleted_at')->ignore($docente->id)],
            'asignaturas'  => ['nullable','array'],
            'enlaces'      => ['nullable','array'],
        ]);

        $docente->update([
            ...$data,
            'asignaturas' => $data['asignaturas'] ?? [],
            'enlaces'     => $data['enlaces'] ?? [],
        ]);

        // 🔹 Sincronizar cambios con su usuario (si existe)
        $user = User::where('email', $docente->getOriginal('email'))->first();
        if ($user) {
            $user->update([
                'name'  => $data['nombre'],
                'email' => $data['email'],
            ]);
        }

        return back()->with('success', 'Docente actualizado correctamente.');
    }

    /**
     * DELETE /admin/docentes/{docente}
     */
    public function destroy(Docente $docente)
    {
        $docente->delete();
        return back()->with('success', 'Docente enviado a la papelera.');
    }

    /**
     * PATCH /admin/docentes/{id}/restore
     */
    public function restore($id)
    {
        $docente = Docente::withTrashed()->findOrFail($id);
        if ($docente->trashed()) {
            $docente->restore();
        }

        return back()->with('success', 'Docente restaurado.');
    }

    /**
     * DELETE /admin/docentes/{id}/force
     */
    public function forceDestroy($id)
    {
        $docente = Docente::withTrashed()->findOrFail($id);
        $docente->forceDelete();

        return back()->with('success', 'Docente eliminado definitivamente.');
    }
}
