<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Carbon\Carbon;

// ✅ Modelos tuyos
use App\Models\Autoridad;
use App\Models\Docente;
use App\Models\Laboratorio;
use App\Models\LineaInvestigacion;
use App\Models\Pasantia;
use App\Models\Pregrado;
use App\Models\Postgrado;
use App\Models\ProyectoInvestigacion;
use App\Models\Publicacion;
use App\Models\TicProject;

class AdminDashboardController extends Controller
{
    public function index(Request $request)
    {
        // Rango por defecto: últimos 12 meses (365 días)
        $to   = Carbon::parse($request->input('to', now()->toDateString()))->endOfDay();
        $from = Carbon::parse($request->input('from', $to->copy()->subDays(364)->toDateString()))->startOfDay();

        // Helper para series mensuales cero-fill
        $makeEmptyMonths = function (Carbon $from, Carbon $to) {
            $cursor = $from->copy()->startOfMonth();
            $end    = $to->copy()->startOfMonth();
            $res = [];
            while ($cursor <= $end) {
                $key = $cursor->format('Y-m');
                $res[$key] = 0;
                $cursor->addMonth();
            }
            return $res;
        };

        $monthlySeries = function ($modelClass, Carbon $from, Carbon $to, $dateCol = 'created_at') use ($makeEmptyMonths) {
            // Si el modelo no tiene la columna, devolvemos arreglo vacío para no fallar
            $instance = new $modelClass;
            if (!Schema::hasColumn($instance->getTable(), $dateCol)) {
                return [];
            }

            $base = $makeEmptyMonths($from, $to);

            $rows = $modelClass::query()
                ->whereBetween($dateCol, [$from, $to])
                ->selectRaw("DATE_FORMAT($dateCol, '%Y-%m') as ym, COUNT(*) as c")
                ->groupBy('ym')
                ->orderBy('ym')
                ->pluck('c', 'ym')
                ->toArray();

            // Merge respetando ceros
            foreach ($rows as $ym => $c) {
                if (array_key_exists($ym, $base)) {
                    $base[$ym] = (int)$c;
                }
            }

            // Salida como labels + data para gráficas
            return [
                'labels' => array_keys($base),
                'data'   => array_values($base),
            ];
        };

        // ========== Totales ==========
        $totals = [
            'autoridades'          => Autoridad::count(),
            'docentes'             => Docente::count(),
            'laboratorios'         => Laboratorio::count(),
            'lineas_investigacion' => LineaInvestigacion::count(),
            'pasantias'            => Pasantia::count(),
            'pregrados'            => Pregrado::count(),
            'postgrados'           => Postgrado::count(),
            'proyectos'            => ProyectoInvestigacion::count(),
            'publicaciones'        => Publicacion::count(),
            'tic_projects'         => TicProject::count(),
        ];

        // ========== Desgloses por estado / flags ==========
        $breakdowns = [];

        // Pasantías por estado
        if (Schema::hasColumn((new Pasantia)->getTable(), 'estado')) {
            $breakdowns['pasantias_por_estado'] = Pasantia::select('estado', DB::raw('COUNT(*) as total'))
                ->groupBy('estado')
                ->orderByDesc('total')
                ->get();
        }

        // Proyectos por estado
        if (Schema::hasColumn((new ProyectoInvestigacion)->getTable(), 'estado')) {
            $breakdowns['proyectos_por_estado'] = ProyectoInvestigacion::select('estado', DB::raw('COUNT(*) as total'))
                ->groupBy('estado')
                ->orderByDesc('total')
                ->get();
        }

        // Publicaciones por activo
        if (Schema::hasColumn((new Publicacion)->getTable(), 'activo')) {
            $breakdowns['publicaciones_activas'] = [
                'activos'   => Publicacion::where('activo', true)->count(),
                'inactivos' => Publicacion::where('activo', false)->count(),
            ];
        }

        // Pregrados visibles
        if (Schema::hasColumn((new Pregrado)->getTable(), 'is_visible')) {
            $breakdowns['pregrados_visibles'] = [
                'visibles'   => Pregrado::where('is_visible', true)->count(),
                'no_visibles'=> Pregrado::where('is_visible', false)->count(),
            ];
        }

        // Postgrados publicados
        if (Schema::hasColumn((new Postgrado)->getTable(), 'published')) {
            $breakdowns['postgrados_publicados'] = [
                'publicados'   => Postgrado::where('published', true)->count(),
                'no_publicados'=> Postgrado::where('published', false)->count(),
            ];
        }

        // ========== Series mensuales (últimos 12 meses) ==========
        $series = [
            'autoridades'          => $monthlySeries(Autoridad::class, $from, $to),
            'docentes'             => $monthlySeries(Docente::class, $from, $to),
            'laboratorios'         => $monthlySeries(Laboratorio::class, $from, $to),
            'lineas_investigacion' => $monthlySeries(LineaInvestigacion::class, $from, $to),
            'pasantias'            => $monthlySeries(Pasantia::class, $from, $to),
            'pregrados'            => $monthlySeries(Pregrado::class, $from, $to),
            'postgrados'           => $monthlySeries(Postgrado::class, $from, $to),
            'proyectos'            => $monthlySeries(ProyectoInvestigacion::class, $from, $to),
            'publicaciones'        => $monthlySeries(Publicacion::class, $from, $to),
            'tic_projects'         => $monthlySeries(TicProject::class, $from, $to),
        ];

        // ========== Tops / nubes simples ==========
        $tops = [];

        // Pasantías por carrera
        if (Schema::hasColumn((new Pasantia)->getTable(), 'carrera')) {
            $tops['pasantias_por_carrera'] = Pasantia::select('carrera', DB::raw('COUNT(*) as total'))
                ->groupBy('carrera')
                ->orderByDesc('total')
                ->limit(8)
                ->get();
        }

        // Proyectos por línea de investigación
        if (Schema::hasColumn((new ProyectoInvestigacion)->getTable(), 'linea_investigacion')) {
            $tops['proyectos_por_linea'] = ProyectoInvestigacion::select('linea_investigacion', DB::raw('COUNT(*) as total'))
                ->groupBy('linea_investigacion')
                ->orderByDesc('total')
                ->limit(8)
                ->get();
        }

        // TicProjects por carrera y por profesor a cargo
        $tic = new TicProject;
        if (Schema::hasColumn($tic->getTable(), 'carrera')) {
            $tops['tic_por_carrera'] = TicProject::select('carrera', DB::raw('COUNT(*) as total'))
                ->groupBy('carrera')
                ->orderByDesc('total')
                ->limit(8)
                ->get();
        }
        if (Schema::hasColumn($tic->getTable(), 'profesor_cargo')) {
            $tops['tic_por_profesor'] = TicProject::select('profesor_cargo', DB::raw('COUNT(*) as total'))
                ->groupBy('profesor_cargo')
                ->orderByDesc('total')
                ->limit(8)
                ->get();
        }

        // ========== Próximos vencimientos (pasantías) ==========
        $upcoming = [];
        $pas = new Pasantia;
        if (Schema::hasColumn($pas->getTable(), 'fecha_maxima_postulacion')) {
            $upcoming['pasantias_vencen_30d'] = Pasantia::whereNotNull('fecha_maxima_postulacion')
                ->whereBetween('fecha_maxima_postulacion', [now(), now()->addDays(30)])
                ->orderBy('fecha_maxima_postulacion')
                ->limit(10)
                ->get(['id','titulo','empresa','carrera','fecha_maxima_postulacion','estado']);
        }

        // ========== Respuesta Inertia ==========
        return inertia('Admin/Dashboard/Index', [
            // Rango (útil para mostrar en el UI del filtro)
            'range' => [
                'from' => $from->toDateString(),
                'to'   => $to->toDateString(),
            ],
            'totals'      => $totals,
            'breakdowns'  => $breakdowns,
            'series'      => $series,
            'tops'        => $tops,
            'upcoming'    => $upcoming,
        ]);
    }
}
