<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Autoridad;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Support\Str;
use Inertia\Inertia;

class AdminAutoridadController extends Controller
{
    private const GRUPO_KEYS = ['jefatura','pregrado','posgrado','administrativo'];
    private const LABELS = [
        'jefatura'      => 'Jefatura del Departamento',
        'pregrado'      => 'Coordinadores de Carrera de Pregrado',
        'posgrado'      => 'Coordinadores de Posgrado',
        'administrativo'=> 'Apoyo Administrativo',
    ];

    /**
     * GET /admin/autoridades
     * Filtros: q, grupo_key, area, trash (none|with|only)
     * Orden: sort (orden|nombre|grupo_key|grupo|email|created_at), dir (asc|desc)
     */
    public function index(Request $request)
    {
        $q         = $request->string('q')->toString();
        $grupoKey  = $request->string('grupo_key')->toString();
        $area      = $request->string('area')->toString();
        $trash     = $request->string('trash', 'none')->toString(); // 'none' | 'with' | 'only'
        $perPage   = (int) max(5, min(100, $request->integer('per_page', 10)));
        $sort      = $request->string('sort', 'orden')->toString(); // 'orden'|...
        $dir       = $request->string('dir', 'asc')->toString();    // 'asc'|'desc'

        $builder = Autoridad::query();

        // Papelera / SoftDeletes
        if ($trash === 'with') {
            $builder->withTrashed();
        } elseif ($trash === 'only') {
            $builder->onlyTrashed();
        }

        // Búsqueda general
        $builder->when($q, function ($query, $search) {
            $query->where(function ($q2) use ($search) {
                $q2->where('slug', 'like', "%{$search}%")
                   ->orWhere('nombre', 'like', "%{$search}%")
                   ->orWhere('cargo', 'like', "%{$search}%")
                   ->orWhere('email', 'like', "%{$search}%")
                   ->orWhere('area', 'like', "%{$search}%")
                   ->orWhere('grupo', 'like', "%{$search}%")
                   ->orWhere('ubicacion', 'like', "%{$search}%")
                   ->orWhere('telefono', 'like', "%{$search}%");
            });
        });

        // Filtro por grupo_key y área
        $builder->when($grupoKey, fn($q) => $q->where('grupo_key', $grupoKey));
        $builder->when($area,     fn($q) => $q->where('area', $area));

        // Orden permitido
        $allowedSorts = ['orden','nombre','grupo_key','grupo','email','created_at'];
        if (!in_array($sort, $allowedSorts, true)) {
            $sort = 'orden';
        }
        $builder->orderBy($sort, $dir === 'desc' ? 'desc' : 'asc')->orderBy('nombre');

        // Paginación + mapeo
        $autoridades = $builder
            ->paginate($perPage)
            ->withQueryString()
            ->through(function (Autoridad $a) {
                return [
                    'id'            => $a->id,
                    'slug'          => $a->slug,
                    'grupo_key'     => $a->grupo_key,
                    'grupo'         => $a->grupo,
                    'nombre'        => $a->nombre,
                    'cargo'         => $a->cargo,
                    'area'          => $a->area,
                    'email'         => $a->email,
                    'telefono'      => $a->telefono,
                    'horario_dias'  => $a->horario_dias,
                    'horario_rango' => $a->horario_rango,
                    'horario'       => $a->horario, // accessor combinado
                    'ubicacion'     => $a->ubicacion,
                    'foto'          => $a->foto,
                    'orden'         => $a->orden,
                    'deleted_at'    => $a->deleted_at?->toISOString(),
                    'created_at'    => $a->created_at?->toISOString(),
                    'updated_at'    => $a->updated_at?->toISOString(),
                ];
            });

        // Datos para filtros dinámicos
        $areas = Autoridad::select('area')->whereNotNull('area')->distinct()->pluck('area')->sort()->values();
        $grupoKeys = self::GRUPO_KEYS;

        return Inertia::render('Admin/Autoridades/Index', [
            'filters'     => compact('q','grupoKey','area','trash','perPage','sort','dir'),
            'areas'       => $areas,
            'grupoKeys'   => $grupoKeys,
            'labels'      => self::LABELS,
            'autoridades' => $autoridades,
        ]);
    }

    /**
     * POST /admin/autoridades
     * Crear autoridad. Sube imagen opcional (foto_file).
     */
    public function store(Request $request)
    {
        $data = $request->validate([
            'slug'          => ['nullable','string','max:255', Rule::unique('autoridades','slug')->whereNull('deleted_at')],
            'grupo_key'     => ['required', Rule::in(self::GRUPO_KEYS)],
            'grupo'         => ['nullable','string','max:255'],
            'nombre'        => ['required','string','max:255'],
            'cargo'         => ['required','string','max:255'],
            'area'          => ['nullable','string','max:255'],
            'email'         => ['nullable','email','max:255', Rule::unique('autoridades','email')->whereNull('deleted_at')],
            'telefono'      => ['nullable','string','max:255'],
            'horario_dias'  => ['nullable','string','max:255'],
            'horario_rango' => ['nullable','string','max:255'],
            'ubicacion'     => ['nullable','string','max:255'],
            'foto'          => ['nullable','string','max:500'],
            'orden'         => ['nullable','integer','min:0'],
            'foto_file'     => ['nullable','image','max:2048'], // 2MB
        ]);

        // Derivar grupo si viene vacío
        if (empty($data['grupo'])) {
            $data['grupo'] = self::LABELS[$data['grupo_key']] ?? $data['grupo_key'];
        }

        // Slug si no llega
        if (empty($data['slug'])) {
            $seed = $data['nombre'].'-'.$data['cargo'].'-'.Str::random(4);
            $slug = Str::slug($seed);
            // asegurar unicidad rápida
            $base = $slug; $i = 1;
            while (Autoridad::withTrashed()->where('slug', $slug)->exists()) {
                $slug = $base.'-'.$i++;
            }
            $data['slug'] = $slug;
        }

        // Subida de imagen
        if ($request->hasFile('foto_file')) {
            $path = $request->file('foto_file')->store('autoridades', 'public');
            $data['foto'] = "/storage/{$path}";
        }
        unset($data['foto_file']);

        // Default orden
        if (!isset($data['orden'])) {
            $data['orden'] = 0;
        }

        Autoridad::create($data);

        return back()->with('success', 'Autoridad creada correctamente.');
    }

    /**
     * PATCH /admin/autoridades/{autoridad}
     * Actualiza autoridad.
     */
    public function update(Request $request, Autoridad $autoridad)
    {
        $data = $request->validate([
            'slug'          => ['nullable','string','max:255', Rule::unique('autoridades','slug')->whereNull('deleted_at')->ignore($autoridad->id)],
            'grupo_key'     => ['required', Rule::in(self::GRUPO_KEYS)],
            'grupo'         => ['nullable','string','max:255'],
            'nombre'        => ['required','string','max:255'],
            'cargo'         => ['required','string','max:255'],
            'area'          => ['nullable','string','max:255'],
            'email'         => ['nullable','email','max:255', Rule::unique('autoridades','email')->whereNull('deleted_at')->ignore($autoridad->id)],
            'telefono'      => ['nullable','string','max:255'],
            'horario_dias'  => ['nullable','string','max:255'],
            'horario_rango' => ['nullable','string','max:255'],
            'ubicacion'     => ['nullable','string','max:255'],
            'foto'          => ['nullable','string','max:500'],
            'orden'         => ['nullable','integer','min:0'],
            'foto_file'     => ['nullable','image','max:2048'],
        ]);

        // Si grupo está vacío, derive del grupo_key
        if (empty($data['grupo'])) {
            $data['grupo'] = self::LABELS[$data['grupo_key']] ?? $data['grupo_key'];
        }

        // Subida de imagen nueva si llega
        if ($request->hasFile('foto_file')) {
            $path = $request->file('foto_file')->store('autoridades', 'public');
            $data['foto'] = "/storage/{$path}";
        }
        unset($data['foto_file']);

        // Slug autogenerado si no tiene
        if (empty($data['slug'])) {
            $seed = $data['nombre'].'-'.$data['cargo'].'-'.Str::random(4);
            $slug = Str::slug($seed);
            $base = $slug; $i = 1;
            while (Autoridad::withTrashed()->where('slug', $slug)->where('id', '!=', $autoridad->id)->exists()) {
                $slug = $base.'-'.$i++;
            }
            $data['slug'] = $slug;
        }

        $autoridad->update($data);

        return back()->with('success', 'Autoridad actualizada correctamente.');
    }

    /**
     * DELETE /admin/autoridades/{autoridad}
     * Soft delete.
     */
    public function destroy(Autoridad $autoridad)
    {
        $autoridad->delete();
        return back()->with('success', 'Autoridad enviada a la papelera.');
    }

    /**
     * PATCH /admin/autoridades/{id}/restore
     * Restaurar.
     */
    public function restore($id)
    {
        $autoridad = Autoridad::withTrashed()->findOrFail($id);
        if ($autoridad->trashed()) {
            $autoridad->restore();
        }
        return back()->with('success', 'Autoridad restaurada.');
    }

    /**
     * DELETE /admin/autoridades/{id}/force
     * Eliminación definitiva.
     */
    public function forceDestroy($id)
    {
        $autoridad = Autoridad::withTrashed()->findOrFail($id);
        $autoridad->forceDelete();
        return back()->with('success', 'Autoridad eliminada definitivamente.');
    }
}
